# bot/envato.py
from pathlib import Path
import re
import time
import requests
from playwright.sync_api import sync_playwright
from .utils import slugify

# سلکتورها (هم با ویدئو، هم بدون ویدئو)
SELECTORS = {
    "h1_title": 'h1, [data-testid="item-title"]',
    # پوشش هر سه حالت: ویدئویی، ایمیج عادی، و کانتینر fallback
    "thumb_container": (
        '[data-testid="pop-out-video-container"], '
        '[data-testid="image-preview"], '
        '[data-testid="default-image-container"]'
    ),
    "fallback_img": 'img[src*="envatousercontent.com"], img[src*="elements-resized"]',
    "video_tag": 'video',

    "download_button": '[data-testid="download-button"], button:has-text("Download")',
    "download_modal": '[data-testid="add-to-project-modal"], [data-testid="download-modal"]',
    "without_license_btn": '[data-testid="download-without-license-button"]',
    "license_and_download_btn": 'button:has-text("License & download")',
    "license_radio_any": 'input[type="radio"]',
}

class EnvatoSession:
    def __init__(self, session_dir: str = "./session", headless: bool = False):
        self.session_dir = session_dir
        self.headless = headless
        Path(session_dir).mkdir(parents=True, exist_ok=True)

    def ensure_login(self):
        with sync_playwright() as p:
            ctx = p.chromium.launch_persistent_context(
                user_data_dir=self.session_dir,
                headless=self.headless,
                args=["--disable-blink-features=AutomationControlled"],
            )
            page = ctx.new_page()
            try:
                page.goto("https://elements.envato.com", timeout=60000)
                page.wait_for_load_state("networkidle", timeout=60000)
            finally:
                ctx.close()

class EnvatoWorker:
    def __init__(self, session_dir, headless, timeout_sec, retries, base_local, msg_q):
        self.session_dir = session_dir
        self.headless = headless
        self.timeout_sec = timeout_sec
        self.retries = retries
        self.base_local = Path(base_local); self.base_local.mkdir(parents=True, exist_ok=True)
        self.msg_q = msg_q

    def _new(self):
        p = sync_playwright().start()
        ctx = p.chromium.launch_persistent_context(
            user_data_dir=self.session_dir,
            headless=self.headless,
            accept_downloads=True,
            args=["--disable-blink-features=AutomationControlled"],
        )
        page = ctx.new_page()
        page.set_default_navigation_timeout(60000)
        page.set_default_timeout(60000)
        return p, ctx, page

    # ---------- helpers ----------
    def _parse_url_from_style(self, style_value: str):
        if not style_value:
            return None
        m = re.search(r'url\(["\']?(.*?)["\']?\)', style_value)
        return m.group(1) if m else None

    def _best_from_srcset(self, srcset: str):
        # آخرین آیتم معمولاً بزرگترین است:  "url1 433w, url2 860w, url3 1016w, ..."
        try:
            last = srcset.split(",")[-1].strip()
            return last.split()[0]
        except Exception:
            return None

    def _extract_thumb_url(self, page):
        """تصویر شاخص را از: CSS --bg یا IMG(src/srcset) برمی‌گرداند."""
        # 1) از CSS variable --bg روی کانتینرهای تعریف‌شده
        try:
            el = page.locator(SELECTORS["thumb_container"]).first
            if el.count() > 0:
                val = el.evaluate(
                    "el => getComputedStyle(el).getPropertyValue('--bg') || el.style.getPropertyValue('--bg')"
                )
                url = self._parse_url_from_style(val)
                if url:
                    return url
        except:
            pass

        # 2) IMG مستقیم داخل همان کانتینر (src یا srcset)
        try:
            img = page.locator(f"{SELECTORS['thumb_container']} img").first
            if img.count() > 0:
                src = img.get_attribute("src")
                if src: return src
                srcset = img.get_attribute("srcset")
                if srcset:
                    best = self._best_from_srcset(srcset)
                    if best: return best
        except:
            pass

        # 3) fallback عمومی: هر <img> با دامنه envatousercontent / elements-resized
        try:
            img2 = page.locator(SELECTORS["fallback_img"]).first
            if img2.count() > 0:
                src = img2.get_attribute("src")
                if src: return src
                srcset = img2.get_attribute("srcset")
                if srcset:
                    best = self._best_from_srcset(srcset)
                    if best: return best
        except:
            pass

        # 4) fallback: هر عنصر style که url داشته باشد
        try:
            cand = page.locator(
                '[style*="envatousercontent.com"], [style*="elements-video-cover-images"], [style*="elements-cover-images"]'
            ).first
            if cand.count() > 0:
                s = cand.get_attribute("style") or ""
                url = self._parse_url_from_style(s)
                if url: return url
        except:
            pass

        return None

    def _download_zip(self, page):
        page.click(SELECTORS["download_button"], timeout=10000)
        page.wait_for_selector(SELECTORS["download_modal"], timeout=15000)
        btn = page.locator(SELECTORS["without_license_btn"]).first
        if btn.count() > 0 and btn.is_visible():
            with page.expect_download(timeout=self.timeout_sec * 1000) as dl_info:
                btn.click()
            return dl_info.value
        radios = page.locator(SELECTORS["license_radio_any"])
        if radios.count() > 0:
            try: radios.first.check(timeout=3000)
            except: radios.first.click()
        with page.expect_download(timeout=self.timeout_sec * 1000) as dl_info:
            page.click(SELECTORS["license_and_download_btn"])
        return dl_info.value

    def _download_binary(self, url, out_path, timeout=60):
        try:
            headers={"User-Agent":"Mozilla/5.0","Referer":"https://elements.envato.com/","Accept":"*/*"}
            with requests.get(url, headers=headers, timeout=timeout, stream=True) as r:
                r.raise_for_status()
                with open(out_path,"wb") as f:
                    for chunk in r.iter_content(262144):
                        if chunk: f.write(chunk)
            return True
        except Exception:
            return False

    # ---------- main ----------
    def process(self, url: str):
        last_err=None
        for i in range(self.retries + 1):
            p, ctx, page = self._new()
            try:
                self.msg_q.put(f"→ Open: {url}")
                page.goto(url, timeout=60000, wait_until="domcontentloaded")
                page.wait_for_selector(SELECTORS["h1_title"], timeout=60000)

                title = page.locator(SELECTORS["h1_title"]).first.inner_text().strip()
                slug = slugify(title)

                item_dir = self.base_local / slug
                item_dir.mkdir(parents=True, exist_ok=True)
                img_path = item_dir / f"{slug}.jpg"
                mp4_path = item_dir / f"{slug}.mp4"
                zip_path = item_dir / f"{slug}.zip"

                thumb_url = self._extract_thumb_url(page)

                video_url=None
                try:
                    v = page.locator(SELECTORS["video_tag"]).first
                    if v and v.count()>0:
                        src=v.get_attribute("src")
                        if src and src.lower().endswith(".mp4"): video_url=src
                except: pass

                image_local=None
                if thumb_url:
                    if self._download_binary(thumb_url, img_path, timeout=60):
                        image_local=str(img_path)

                video_local=None
                if video_url:
                    if self._download_binary(video_url, mp4_path, timeout=120):
                        video_local=str(mp4_path)

                d = self._download_zip(page)
                d.save_as(str(zip_path))
                time.sleep(1)

                ctx.close(); p.stop()
                return {
                    "url":url,
                    "title":title,
                    "slug":slug,
                    "local":{"zip":str(zip_path),"image":image_local,"video":video_local},
                    "extracted":{"thumb_url":thumb_url,"video_url":video_url},
                }

            except Exception as e:
                last_err=e
                self.msg_q.put(f"retry {i+1}/{self.retries} due to: {e}")
                time.sleep(1.5)
            finally:
                try: ctx.close(); p.stop()
                except: pass
        raise last_err
