import tkinter as tk
from tkinter import ttk, messagebox
import threading, queue, configparser
from pathlib import Path

from bot.envato import EnvatoSession, EnvatoWorker
from bot.ftp_client import FtpClient
from bot.api_client import EditoraAPI
from bot.utils import read_urls, setup_logging

class App(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title("Editora Envato Bot")
        self.geometry("820x560")
        self.running = False
        self.msg_q = queue.Queue()
        self.cfg = configparser.ConfigParser()
        self.cfg.read("config.ini", encoding="utf-8")
        self._ui()
        self.after(200, self._drain)

    def _ui(self):
        root = ttk.Frame(self, padding=12); root.pack(fill="both", expand=True)
        row = ttk.Frame(root); row.pack(fill="x", pady=4)
        ttk.Label(row, text="Year:").pack(side="left")
        self.var_year = tk.StringVar(value=self.cfg.get("behavior","default_year",fallback="1404"))
        ttk.Entry(row, textvariable=self.var_year, width=8).pack(side="left", padx=8)
        ttk.Label(row, text="Month:").pack(side="left")
        self.var_month = tk.StringVar(value=self.cfg.get("behavior","default_month",fallback="8"))
        ttk.Entry(row, textvariable=self.var_month, width=5).pack(side="left", padx=8)

        ctrl = ttk.Frame(root); ctrl.pack(fill="x", pady=6)
        ttk.Button(ctrl, text="Login to Envato", command=self.login).pack(side="left")
        self.btn_start = ttk.Button(ctrl, text="Start", command=self.start); self.btn_start.pack(side="left", padx=8)
        self.btn_stop  = ttk.Button(ctrl, text="Stop", command=self.stop, state="disabled"); self.btn_stop.pack(side="left")

        self.status = tk.StringVar(value="Ready."); ttk.Label(root, textvariable=self.status).pack(anchor="w", pady=(8,2))
        self.prog = ttk.Progressbar(root, mode="determinate", maximum=100, value=0); self.prog.pack(fill="x", pady=(0,8))
        self.box = tk.Listbox(root, height=20); self.box.pack(fill="both", expand=True)

        ttk.Label(root, text="urls.txt کنار برنامه است. لاگ: ./logs/bot.log").pack(anchor="w", pady=4)

    def _drain(self):
        try:
            while True:
                m = self.msg_q.get_nowait()
                self.box.insert("end", m); self.box.see("end")
        except queue.Empty:
            pass
        self.after(250, self._drain)

    def login(self):
        EnvatoSession(session_dir=self.cfg.get("envato","session_dir",fallback="./session"),
                      headless=self.cfg.getboolean("behavior","headless",fallback=False)).ensure_login()
        messagebox.showinfo("Login", "Session saved.")

    def start(self):
        if self.running: return
        y, m = self.var_year.get().strip(), self.var_month.get().strip()
        if not (y.isdigit() and m.isdigit()): 
            messagebox.showerror("Error","Year/Month must be numbers."); return
        self.running = True; self.btn_start.config(state="disabled"); self.btn_stop.config(state="normal")
        threading.Thread(target=self._run, args=(y,m), daemon=True).start()

    def stop(self):
        self.running = False
        self.status.set("Stopping...")

    def _run(self, year, month):
        setup_logging(Path(self.cfg.get("paths","logs_dir",fallback="./logs"))/"bot.log")
        urls = read_urls("urls.txt")
        if not urls:
            self.msg_q.put("No URLs in urls.txt"); self._finish(); return
        self.prog.config(maximum=len(urls), value=0)

        env = EnvatoWorker(
            session_dir=self.cfg.get("envato","session_dir",fallback="./session"),
            headless=self.cfg.getboolean("behavior","headless",fallback=False),
            timeout_sec=self.cfg.getint("behavior","timeout_sec",fallback=20),
            retries=self.cfg.getint("behavior","retries",fallback=2),
            base_local=Path(self.cfg.get("paths","base_local",fallback="./data")),
            msg_q=self.msg_q
        )
        ftp = FtpClient(self.cfg.get("ftp","host"), self.cfg.get("ftp","user"), self.cfg.get("ftp","pass"), msg_q=self.msg_q)
        api = EditoraAPI(self.cfg.get("wp","base_url"), self.cfg.get("wp","jwt_token"), self.cfg.get("wp","add_product_endpoint"), msg_q=self.msg_q)

        done = 0
        for url in urls:
            if not self.running: break
            try:
                self.status.set(f"Processing: {url}")
                item = env.process(url)

                remote = ftp.upload_all(
                    year=year, month=month,
                    basename=item["slug"],
                    local_img=item["local"].get("image"),
                    local_mp4=item["local"].get("video"),
                    local_zip=item["local"].get("zip")
                )
                cdn = ftp.to_cdn(remote)

                cover_url = cdn.get("IMG_URL") or ""
                preview_url = cdn.get("VIDEO_URL") or ""

                api_id = api.add_product(
                    title=item["title"],
                    cover=cover_url,
                    preview=preview_url,
                    download_link=cdn["ZIP_URL"],
                    envato_url=item["url"]
                )
                self.msg_q.put(f"✓ Published (ID {api_id})")
            except Exception as e:
                import logging
                logging.error(f"Error {url}: {e}")
                self.msg_q.put(f"❌ {e}")
            done += 1
            self.prog.config(value=done)
        self._finish()

    def _finish(self):
        self.running = False
        self.btn_start.config(state="normal"); self.btn_stop.config(state="disabled")
        self.status.set("Done.")

if __name__ == "__main__":
    App().mainloop()
